# Step-by-Step Server Testing Guide

## Prerequisites

### 1. System Requirements
- **OS**: Linux (Ubuntu 20.04+ recommended) or macOS
- **Python**: Python 3.8 or higher
- **RAM**: Minimum 8GB (16GB+ recommended for large documents)
- **Disk Space**: At least 10GB free (for models and data)
- **CPU**: Multi-core processor recommended

### 2. Install Python and pip

#### For Ubuntu/Debian:
```bash
# Check Python version
python3 --version

# If not installed, install Python 3.8+
sudo apt update
sudo apt install python3 python3-pip python3-venv -y
```

#### For RHEL/CentOS/Fedora (dnf):
```bash
# Check Python version
python3 --version

# If not installed, install Python 3.8+
# First, try to fix repository issues (see troubleshooting below)
sudo dnf clean all
sudo dnf makecache

# If EA4 repository is causing issues, disable it temporarily:
sudo dnf config-manager --disable EA4-a10

# Then install Python
sudo dnf install python3 python3-pip python3-venv -y
```

**Note:** If you encounter repository errors, see the [Repository Troubleshooting](#repository-troubleshooting) section below.

---

## Step 1: Set Up Project Directory

```bash
# Navigate to your project directory
cd /path/to/apple

# Verify project structure
ls -la
# Should see: financial_report_gen.py, requirements.txt, etc.
```

---

## Step 2: Install and Configure Ollama

### 2.1 Install Ollama
```bash
# Download and install Ollama
curl -fsSL https://ollama.com/install.sh | sh

# Verify installation
ollama --version
```

### 2.2 Start Ollama Service
```bash
# Start Ollama service (if not running)
ollama serve

# In a new terminal, verify it's running
curl http://localhost:11434/api/tags
```

### 2.3 Pull Required Model
```bash
# Pull the llama3.1:8b model (this may take 10-20 minutes depending on connection)
ollama pull llama3.1:8b

# Verify model is available
ollama list
# Should show: llama3.1:8b
```

### 2.4 Test Ollama Connection
```bash
# Test if Ollama is responding
ollama run llama3.1:8b "Hello, can you respond?"
# If you get a response, Ollama is working correctly
```

### 2.5 Remove Ollama Models (if needed)
```bash
# List all installed models
ollama list

# Remove a specific model
ollama rm llama3.1:8b

# Remove multiple models
ollama rm model1 model2 model3

# Verify removal
ollama list
```

**Note:** Models can take up significant disk space (4-40GB+). Removing unused models can free up space.

---

## Step 3: Set Up Python Environment

### 3.1 Create Virtual Environment
```bash
# Create virtual environment
python3 -m venv venv

# Activate virtual environment
source venv/bin/activate

# Verify activation (prompt should show (venv))
which python
```

### 3.2 Install Python Dependencies
```bash
# Upgrade pip first
pip install --upgrade pip

# Install all requirements
pip install -r requirements.txt

# Verify installations
pip list | grep -E "llama-index|faiss|reportlab|pypdf"
```

**Expected output should show:**
- llama-index-core
- llama-index-llms-ollama
- llama-index-embeddings-huggingface
- llama-index-vector-stores-faiss
- faiss-cpu
- reportlab
- pypdf

---

## Step 4: Prepare Data Directory

### 4.1 Create Data Folder
```bash
# Create the Niraj_Upload directory (as expected by the script)
mkdir -p Niraj_Upload

# Verify it exists
ls -la Niraj_Upload/
```

### 4.2 Add Financial Documents
```bash
# Copy your financial documents to Niraj_Upload folder
# The script will recursively read all files in this directory

# Example: Copy PDF files
cp /path/to/your/financial/*.pdf Niraj_Upload/

# Or copy entire directory structure
cp -r /path/to/financial_data/* Niraj_Upload/

# Verify files are present
find Niraj_Upload -type f | head -10
```

**Supported file types:**
- PDF files (.pdf)
- Text files (.txt)
- Markdown files (.md)
- Other formats supported by LlamaIndex SimpleDirectoryReader

---

## Step 5: Test Script Execution

### 5.1 Quick Test (Verify Setup)
```bash
# Activate virtual environment (if not already active)
source venv/bin/activate

# Run a quick syntax check
python3 -m py_compile financial_report_gen.py

# If no errors, proceed to full test
```

### 5.2 Run the Script
```bash
# Make sure Ollama is running in another terminal
# ollama serve

# Run the script
python3 financial_report_gen.py
```

**Expected output:**
```
📂 Loading documents from Niraj_Upload folder...
✅ Loaded X documents successfully
📊 Creating FAISS vector index from documents...
...
```

### 5.3 Monitor Progress
The script will:
1. Load documents (1-2 minutes)
2. Build/load FAISS index (5-15 minutes first time, faster on subsequent runs)
3. Generate 14 sections in parallel (30-60 minutes)
4. Combine sections into text file
5. Generate PDF report

**Total time:** 45-90 minutes depending on:
- Number of documents
- Server performance
- Ollama response time

---

## Step 6: Verify Output Files

### 6.1 Check Generated Files
```bash
# After script completion, verify output files
ls -lh financial_report*

# Expected files:
# - financial_report_combined.txt (text version)
# - financial_report.pdf (PDF version)
```

### 6.2 Verify PDF
```bash
# Check PDF file size (should be > 1MB for comprehensive report)
ls -lh financial_report.pdf

# If you have a PDF viewer on server, test it
# file financial_report.pdf
# Should show: PDF document
```

---

## Step 7: Troubleshooting Common Issues

### Issue 1: Ollama Connection Error
**Error:** `Connection refused` or `Cannot connect to Ollama`

**Solution:**
```bash
# Check if Ollama is running
ps aux | grep ollama

# Start Ollama if not running
ollama serve

# Test connection
curl http://localhost:11434/api/tags
```

### Issue 2: Model Not Found
**Error:** `model 'llama3.1:8b' not found`

**Solution:**
```bash
# Pull the model
ollama pull llama3.1:8b

# Verify
ollama list
```

### Issue 3: Out of Memory
**Error:** `CUDA out of memory` or system becomes unresponsive

**Solution:**
- Reduce parallel workers in script (line 570: change `max_workers=3` to `max_workers=1`)
- Use a smaller model: `ollama pull llama3.1:8b` (already using smallest)
- Close other applications
- Consider using a server with more RAM

### Issue 4: FAISS Index Errors
**Error:** `Index structure mismatch` or `KeyError`

**Solution:**
```bash
# Delete corrupted index and rebuild
rm -rf faiss_index/

# Re-run script (it will rebuild index automatically)
python3 financial_report_gen.py
```

### Issue 5: Missing Dependencies
**Error:** `ModuleNotFoundError: No module named 'X'`

**Solution:**
```bash
# Reinstall requirements
pip install -r requirements.txt --force-reinstall

# If specific package fails, install individually
pip install llama-index-core
pip install llama-index-llms-ollama
pip install llama-index-embeddings-huggingface
pip install llama-index-vector-stores-faiss
pip install faiss-cpu
pip install reportlab
pip install pypdf
```

### Issue 6: Permission Errors
**Error:** `Permission denied` when creating files

**Solution:**
```bash
# Check directory permissions
ls -la

# Fix permissions if needed
chmod 755 .
chmod 644 *.py
```

### Issue 7: Timeout Errors
**Error:** `Request timeout` or `Connection timeout`

**Solution:**
- The script already has `request_timeout=3600` (60 minutes)
- If still timing out, check Ollama logs
- Consider running sections sequentially instead of parallel

### Issue 8: Repository Errors (dnf/yum)
**Error:** `Failed to download metadata for repo` or `Status code: 404`

**Solution for EA4 repository errors:**
```bash
# Option 1: Disable the problematic repository temporarily
sudo dnf config-manager --disable EA4-a10

# Then retry installation
sudo dnf install python3 python3-pip python3-venv -y
```

**Solution for general repository issues:**
```bash
# Clean all cached metadata
sudo dnf clean all

# Remove problematic repository cache
sudo rm -rf /var/cache/dnf/*

# Rebuild cache
sudo dnf makecache

# If still failing, try with --skip-broken flag
sudo dnf install python3 python3-pip python3-venv -y --skip-broken
```

**Alternative: Install Python from source (if repositories are completely broken):**
```bash
# Install build dependencies
sudo dnf groupinstall "Development Tools" -y
sudo dnf install openssl-devel bzip2-devel libffi-devel zlib-devel -y

# Download Python source
cd /tmp
wget https://www.python.org/ftp/python/3.11.0/Python-3.11.0.tgz
tar xzf Python-3.11.0.tgz
cd Python-3.11.0

# Configure and install
./configure --enable-optimizations
make altinstall

# Verify installation
python3.11 --version
```

**Note:** If Python is already installed, you can skip repository installation and proceed directly to creating the virtual environment.

### Issue 9: No Space Left on Device
**Error:** `OSError: [Errno 28] No space left on device` during pip install

**Solution:**
```bash
# Step 1: Check disk space
df -h

# Step 2: Free up space (choose options based on your needs)

# Option A: Remove unused Ollama models (can free 4-40GB+)
ollama list
ollama rm unused_model_name

# Option B: Clean pip cache
pip cache purge

# Option C: Remove Python cache files
find . -type d -name __pycache__ -exec rm -r {} + 2>/dev/null
find . -type f -name "*.pyc" -delete

# Option D: Clean system logs (requires sudo)
sudo journalctl --vacuum-time=3d

# Option E: Clean package manager cache
sudo dnf clean all  # For RHEL/CentOS/Fedora
# or
sudo apt clean      # For Ubuntu/Debian

# Step 3: Verify space is freed
df -h

# Step 4: Retry installation
pip install -r requirements.txt
```

**Alternative: Install to different location with more space:**
```bash
# Create venv in location with more space
python3 -m venv /path/to/larger/disk/venv
source /path/to/larger/disk/venv/bin/activate
pip install -r requirements.txt
```

**See `FIX_DISK_SPACE.md` for detailed disk space management guide.**

---

## Step 8: Running in Background (Production)

### 8.1 Using nohup
```bash
# Run script in background with output logging
nohup python3 financial_report_gen.py > report_generation.log 2>&1 &

# Check process
ps aux | grep financial_report_gen

# Monitor progress
tail -f report_generation.log
```

### 8.2 Using screen (Recommended)
```bash
# Install screen if not available
sudo apt install screen -y

# Start a screen session
screen -S report_gen

# Run script
python3 financial_report_gen.py

# Detach: Press Ctrl+A, then D
# Reattach: screen -r report_gen
```

### 8.3 Using tmux
```bash
# Install tmux if not available
sudo apt install tmux -y

# Start tmux session
tmux new -s report_gen

# Run script
python3 financial_report_gen.py

# Detach: Press Ctrl+B, then D
# Reattach: tmux attach -t report_gen
```

---

## Step 9: Performance Optimization

### 9.1 For Faster Processing
- **Use GPU**: Install `faiss-gpu` instead of `faiss-cpu` (requires CUDA)
- **Increase Workers**: Change `max_workers=3` to `max_workers=5` (if RAM allows)
- **Use Larger Model**: `ollama pull llama3.1:70b` (requires more RAM)

### 9.2 For Lower Resource Usage
- **Reduce Workers**: Change `max_workers=3` to `max_workers=1`
- **Reduce Top K**: Change `similarity_top_k=20` to `similarity_top_k=10`
- **Use Smaller Model**: Already using smallest (llama3.1:8b)

---

## Step 10: Validation Checklist

Before considering the test successful, verify:

- [ ] Ollama is running and responding
- [ ] Model `llama3.1:8b` is available
- [ ] All Python dependencies installed
- [ ] `Niraj_Upload` folder contains documents
- [ ] Script runs without errors
- [ ] FAISS index is created/loaded successfully
- [ ] All 14 sections are generated
- [ ] `financial_report_combined.txt` is created
- [ ] `financial_report.pdf` is generated and opens correctly
- [ ] PDF contains all sections with proper formatting

---

## Quick Start Command Summary

```bash
# 1. Install Ollama and pull model
curl -fsSL https://ollama.com/install.sh | sh
ollama pull llama3.1:8b

# 2. Set up Python environment
python3 -m venv venv
source venv/bin/activate
pip install -r requirements.txt

# 3. Prepare data
mkdir -p Niraj_Upload
# Copy your documents to Niraj_Upload/

# 4. Start Ollama (in separate terminal)
ollama serve

# 5. Run script
python3 financial_report_gen.py
```

---

## Support and Logs

### View Script Output
```bash
# If running in background, check logs
tail -f report_generation.log

# Or check nohup output
tail -f nohup.out
```

### Check Ollama Logs
```bash
# Ollama logs location (varies by OS)
# Linux: ~/.ollama/logs/
# macOS: ~/.ollama/logs/

# View recent logs
tail -f ~/.ollama/logs/server.log
```

### System Resource Monitoring
```bash
# Monitor CPU and memory usage
top
# or
htop

# Monitor disk space
df -h

# Monitor Python process
ps aux | grep python
```

---

## Next Steps After Successful Test

1. **Schedule Regular Runs**: Set up cron job for automated report generation
2. **Optimize Performance**: Adjust workers and model based on server capacity
3. **Add Error Handling**: Implement retry logic for failed sections
4. **Set Up Monitoring**: Monitor disk space, memory, and Ollama service
5. **Backup Reports**: Archive generated reports regularly

---

## Notes

- First run will be slower (index building)
- Subsequent runs will be faster (index reuse)
- Ensure Ollama service stays running during script execution
- Script automatically handles index corruption and rebuilds if needed
- Total execution time: 45-90 minutes for full report

