#!/bin/bash

# Server Setup Script for Financial Report Generator
# This script automates the initial setup process

set -e  # Exit on error

echo "=========================================="
echo "Financial Report Generator - Server Setup"
echo "=========================================="
echo ""

# Colors for output
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Function to print colored output
print_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

print_error() {
    echo -e "${RED}❌ $1${NC}"
}

# Step 1: Check Python
echo "Step 1: Checking Python installation..."
if command -v python3 &> /dev/null; then
    PYTHON_VERSION=$(python3 --version)
    print_success "Python found: $PYTHON_VERSION"
else
    print_error "Python 3 not found. Please install Python 3.8+ first."
    exit 1
fi

# Step 2: Create virtual environment
echo ""
echo "Step 2: Setting up Python virtual environment..."
if [ ! -d "venv" ]; then
    python3 -m venv venv
    print_success "Virtual environment created"
else
    print_warning "Virtual environment already exists"
fi

# Step 3: Activate virtual environment and install dependencies
echo ""
echo "Step 3: Installing Python dependencies..."
source venv/bin/activate
pip install --upgrade pip --quiet
pip install -r requirements.txt
print_success "Dependencies installed"

# Step 4: Check Ollama
echo ""
echo "Step 4: Checking Ollama installation..."
if command -v ollama &> /dev/null; then
    OLLAMA_VERSION=$(ollama --version)
    print_success "Ollama found: $OLLAMA_VERSION"
    
    # Check if Ollama service is running
    if curl -s http://localhost:11434/api/tags > /dev/null 2>&1; then
        print_success "Ollama service is running"
    else
        print_warning "Ollama service is not running. Start it with: ollama serve"
    fi
    
    # Check if model is available
    if ollama list | grep -q "llama3.1:8b"; then
        print_success "Model llama3.1:8b is available"
    else
        print_warning "Model llama3.1:8b not found. Install with: ollama pull llama3.1:8b"
    fi
else
    print_warning "Ollama not found. Install with: curl -fsSL https://ollama.com/install.sh | sh"
fi

# Step 5: Create data directory
echo ""
echo "Step 5: Setting up data directory..."
if [ ! -d "Niraj_Upload" ]; then
    mkdir -p Niraj_Upload
    print_success "Created Niraj_Upload directory"
    print_warning "Please add your financial documents to Niraj_Upload/ folder"
else
    FILE_COUNT=$(find Niraj_Upload -type f 2>/dev/null | wc -l)
    if [ "$FILE_COUNT" -gt 0 ]; then
        print_success "Niraj_Upload directory exists with $FILE_COUNT file(s)"
    else
        print_warning "Niraj_Upload directory exists but is empty. Add your documents."
    fi
fi

# Step 6: Check FAISS index
echo ""
echo "Step 6: Checking FAISS index..."
if [ -d "faiss_index" ]; then
    print_success "FAISS index directory exists"
else
    print_warning "FAISS index will be created on first run"
fi

# Summary
echo ""
echo "=========================================="
echo "Setup Summary"
echo "=========================================="
echo ""
echo "✅ Python environment: Ready"
echo "✅ Dependencies: Installed"
if command -v ollama &> /dev/null; then
    echo "✅ Ollama: Installed"
    if curl -s http://localhost:11434/api/tags > /dev/null 2>&1; then
        echo "✅ Ollama service: Running"
    else
        echo "⚠️  Ollama service: Not running (start with: ollama serve)"
    fi
    if ollama list | grep -q "llama3.1:8b"; then
        echo "✅ Model: Available"
    else
        echo "⚠️  Model: Not installed (install with: ollama pull llama3.1:8b)"
    fi
else
    echo "⚠️  Ollama: Not installed"
fi

if [ -d "Niraj_Upload" ] && [ "$(find Niraj_Upload -type f 2>/dev/null | wc -l)" -gt 0 ]; then
    echo "✅ Data directory: Ready"
else
    echo "⚠️  Data directory: Empty (add documents to Niraj_Upload/)"
fi

echo ""
echo "=========================================="
echo "Next Steps:"
echo "=========================================="
echo ""
echo "1. If Ollama is not running, start it:"
echo "   ollama serve"
echo ""
echo "2. If model is not installed, pull it:"
echo "   ollama pull llama3.1:8b"
echo ""
echo "3. Add your financial documents to:"
echo "   Niraj_Upload/"
echo ""
echo "4. Activate virtual environment and run:"
echo "   source venv/bin/activate"
echo "   python3 financial_report_gen.py"
echo ""
echo "=========================================="



